#!/usr/bin/python
#
# PyAgeModule (Python Script for GEDitCOM II)
# Created: 2024-03-16 21:22:50 +0000
# Author: John A. Nairn
#
# Description: Subroutines used by the Generation Ages Python.py
#   scripting tutorial

from GEDitCOMII import *

# Collect data for the generation ages report
def CollectAges(famList):
    global numHusbAge,sumHusbAge,numFathAge,sumFathAge
    global numWifeAge,sumWifeAge,numMothAge,sumMothAge

    # initialize counters
    numHusbAge=sumHusbAge=numFathAge=sumFathAge=0
    numWifeAge=sumWifeAge=numMothAge=sumMothAge=0
    
    # progress reporting interval
    fractionStepSize=nextFraction=0.01
    numFams=len(famList)
    for (i,fam) in enumerate(famList):
        # read family record information
        husbRef = fam.husband()
        wifeRef = fam.wife()
        chilList = fam.children()
        mdate = fam.marriageSDN()
        
        # read parent birthdates
        hbdate = wbdate = 0
        if husbRef != "":
            hbdate = husbRef.birthSDN()
        if wifeRef != "":
            wbdate = wifeRef.birthSDN()
        
        # spouse ages at marriage
        if mdate>0:
            if hbdate>0:
                sumHusbAge = sumHusbAge + GetAgeSpan(hbdate,mdate)
                numHusbAge = numHusbAge+1
           
            if wbdate>0:
                sumWifeAge = sumWifeAge + GetAgeSpan(wbdate,mdate)
                numWifeAge = numWifeAge+1
                
        # spouse ages when children were born
        if hbdate > 0 or wbdate > 0:
            for chilRef in chilList:
                cbdate = chilRef.birthSDN()
                if cbdate > 0 and hbdate > 0:
                    sumFathAge = sumFathAge + GetAgeSpan(hbdate,cbdate)
                    numFathAge = numFathAge + 1
                if cbdate > 0 and wbdate > 0:
                    sumMothAge = sumMothAge + GetAgeSpan(wbdate,cbdate)
                    numMothAge = numMothAge + 1
                    
        # time for progress
        fractionDone = float(i+1)/float(numFams)
        if fractionDone > nextFraction:
            ProgressMessage(fractionDone)
            nextFraction = nextFraction+fractionStepSize

# Write the results now in the global variables to a
# GEDitCOM II report using <html> style
def WriteToReport(sname,fname):
    # begin report with <h1> for title
    rpt = ScriptOutput(sname,"html")
    rpt.out("<h1>Generational Age Analysis in " + fname + "</h1>\n")

    # start <table> and give it a <caption> and add header, start body
    rpt.out(MakeTable("begin","caption","Summary of spouse ages when married and when children were born"))
    rpt.out(MakeTable("head",["Age Item","Husband","Wife"],"body"))
    
    # rows for ages when married and when children were borm
    InsertRow("Avg. Age at Marriage", numHusbAge, sumHusbAge, numWifeAge, sumWifeAge, rpt)
    InsertRow("Avg. Age at Childbirth", numFathAge, sumFathAge, numMothAge, sumMothAge, rpt)

    # end the <tbody> and <table> elements
    rpt.out(MakeTable("endbody","end"))
    
    # write out the report
    rpt.write()

# Insert table row with husband and wife results
def InsertRow(rowLabel, numHusb, sumHusb, numWife, sumWife, rpt):
    tr = [rowLabel]
    if numHusb > 0:
        tr.append("{0:.2f}".format(sumHusb / numHusb))
    else:
        tr.append("-")
    if numWife > 0:
        tr.append("{0:.2f}".format(sumWife / numWife))
    else:
        tr.append("-")
    rpt.out(MakeTable("row l r r",tr))
